/** @file   weather.cpp
 * @brief   Implementation of Weather class
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/08/13 10:50:23 $
 * @author  Tomi Lamminsaari
 */

#include "weather.h"
#include "soundsamples.h"
#include <allegro.h>
#include "eng2d.h"
using namespace eng2d;


namespace WeWantWar {


const int Weather::MAX_RAINDROPS;
const int Weather::RAIN;
const int Weather::THUNDER;

float Weather::MAX_X = 8000;
float Weather::MAX_Y = 5000;

int Weather::MINIMUM_THUNDER_DELAY = 500;
int Weather::WIND_ADJUST_DELAY = 120;
float Weather::MAX_WIND_MAGNITUDE = 30.0;



Weather::Weather( int events ) :
  m_events( events ),
  m_thunderCounter( 0 ),
  m_doLightning( false ),
  m_windDirection( 0 ),
  m_windMagnitude( 1 ),
  m_windCounter( 0 )
{
  // Init Rain-effect.
  for (int i=0; i < MAX_RAINDROPS; i++) {
    m_raindrops[i] = new RainDrop;
  }
  this->initRainDrops();
  
  // Init Thunder-effect.
  m_thunderCounter = MINIMUM_THUNDER_DELAY + (rand() % MINIMUM_THUNDER_DELAY);
}



/** Destructor.
 */
Weather::~Weather()
{
  for (int i=0; i < MAX_RAINDROPS; i++) {
    delete m_raindrops[i];
    m_raindrops[i] = 0;
  }
}



void Weather::update()
{
  // Is the rain-effect on
  if ( m_events & Weather::RAIN ) {
    // First we adjust the wind
    m_windCounter -= 1;
    if ( m_windCounter < 0 ) {
      m_windCounter = 10 + ( rand() % WIND_ADJUST_DELAY );
      m_windDirection += ( rand() % 20 ) - 10;
      m_windDirection = m_windDirection % 255;
      float wm = static_cast<float>( rand() % 100 ) / 100.0;
      m_windMagnitude = wm * MAX_WIND_MAGNITUDE;
    }
    
    eng2d::Vec2D windVec(0, -m_windMagnitude );
    windVec.rotate( m_windDirection );

    // And then we move the raindrops by the wind.
    for (int i=0; i < MAX_RAINDROPS; i++) {
      m_raindrops[i]->posX += windVec.x();
      m_raindrops[i]->posY += windVec.y();
      if ( m_raindrops[i]->posX > MAX_X/2 ) {
        m_raindrops[i]->posX -= MAX_X;
      } else if ( m_raindrops[i]->posX < (-MAX_X / 2) ) {
        m_raindrops[i]->posX += MAX_X;
      }
      
      if ( m_raindrops[i]->posY > MAX_Y/2 ) {
        m_raindrops[i]->posY -= MAX_Y;
      } else if ( m_raindrops[i]->posY < (-MAX_Y / 2) ) {
        m_raindrops[i]->posY += MAX_Y;
      }
    
      m_raindrops[i]->altitude += 0.1;
      if ( m_raindrops[i]->altitude > 10 ) {
        m_raindrops[i]->altitude = 3;
      }
    }
  }
  
  // Is the thunder effect on
  if ( m_events & Weather::THUNDER ) {
    m_thunderCounter -= 1;
    if ( m_thunderCounter < 0 ) {
      m_thunderCounter = MINIMUM_THUNDER_DELAY + (rand() % MINIMUM_THUNDER_DELAY);
      m_doLightning = true;
      Sound::playSample(SMP_THUNDER, false);
    }
  }
}



void Weather::redraw()
{
  if ( m_events & RAIN ) {
    int c1 = makecol( 160, 160, 160 );
    int c2 = makecol( 210, 210, 210 );
    int c3 = makecol( 240, 240, 240 );
  
    int x;
    int y;
    drawing_mode(DRAW_MODE_TRANS, 0, 0, 0);
    for (int i=0; i < MAX_RAINDROPS; i++) {
      x = static_cast<int>( m_raindrops[i]->posX / m_raindrops[i]->altitude ) + 320;
      y = static_cast<int>( m_raindrops[i]->posY / m_raindrops[i]->altitude ) + 240;
    
      if ( m_raindrops[i]->altitude < 3 ) {
        set_trans_blender(0,0,0, 120);
        putpixel( Display::buffer, x, y, c1 );
      } else if ( m_raindrops[i]->altitude < 7 ) {
        set_trans_blender(0,0,0, 160);
        putpixel( Display::buffer, x, y, c2 );
      } else {
        set_trans_blender(0,0,0, 200);
        putpixel( Display::buffer, x, y, c3 );
      }
    }
  }
  solid_mode();
  
  if ( m_doLightning ) {
    clear_to_color( Display::buffer, makecol(230,230,230) );
    m_doLightning = false;
  }
}



void Weather::initRainDrops()
{
  for (int i=0; i < MAX_RAINDROPS; i++) {
    m_raindrops[i]->altitude = rand() % 10;
    m_raindrops[i]->posX = (rand() % static_cast<int>(MAX_X)) - MAX_X/2;
    m_raindrops[i]->posY = (rand() % static_cast<int>(MAX_Y)) - MAX_Y/2;
  }
}


} // end of namespace
